<?php
// =======================
// Verify Zarinpal Payment
// =======================

// جلوگیری از اجرای دوباره
if (php_sapi_name() === 'cli') {
    exit;
}

// -----------------------
// Includes
// -----------------------
require_once "../config/db.php";
require_once "../config/payment.php";
require_once "../config/bot.php";

// -----------------------
// Helper: Send Telegram Message
// -----------------------
function sendTelegramMessage($chat_id, $text)
{
    $url = API_URL . "sendMessage";
    $params = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    @file_get_contents($url . '?' . http_build_query($params));
}

// -----------------------
// Validate Input
// -----------------------
$order_id  = isset($_GET['order']) ? intval($_GET['order']) : 0;
$status    = $_GET['Status'] ?? '';
$authority = $_GET['Authority'] ?? '';

if ($order_id <= 0 || empty($authority)) {
    die("Invalid Request");
}

if ($status !== 'OK') {
    die("❌ پرداخت توسط کاربر لغو شد.");
}

// -----------------------
// Fetch Order
// -----------------------
$stmt = $pdo->prepare("
    SELECT 
        o.id,
        o.total_price,
        o.status,
        u.telegram_id
    FROM orders o
    JOIN users u ON o.user_id = u.id
    WHERE o.id = ?
    LIMIT 1
");
$stmt->execute([$order_id]);
$order = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$order) {
    die("Order not found");
}

if ($order['status'] === 'paid') {
    die("این سفارش قبلاً پرداخت شده است.");
}

// -----------------------
// Verify Payment with Zarinpal
// -----------------------
$verifyData = [
    "merchant_id" => ZARINPAL_MERCHANT,
    "amount"      => (int)$order['total_price'],
    "authority"   => $authority
];

$ch = curl_init("https://api.zarinpal.com/pg/v4/payment/verify.json");
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => json_encode($verifyData),
    CURLOPT_HTTPHEADER     => [
        'Content-Type: application/json',
        'Content-Length: ' . strlen(json_encode($verifyData))
    ],
    CURLOPT_TIMEOUT => 30
]);

$response = curl_exec($ch);
curl_close($ch);

$result = json_decode($response, true);

// -----------------------
// Check Verification Result
// -----------------------
if (!isset($result['data']['code']) || $result['data']['code'] != 100) {
    die("❌ پرداخت ناموفق یا تایید نشد.");
}

$ref_id = $result['data']['ref_id'] ?? 'N/A';

// -----------------------
// Transaction (DB Safe)
// -----------------------
try {
    $pdo->beginTransaction();

    // Update order status
    $stmt = $pdo->prepare("
        UPDATE orders 
        SET status = 'paid'
        WHERE id = ?
    ");
    $stmt->execute([$order_id]);

    // Insert payment record
    $stmt = $pdo->prepare("
        INSERT INTO payments (order_id, method, status, ref_id)
        VALUES (?, 'zarinpal', 'success', ?)
    ");
    $stmt->execute([$order_id, $ref_id]);

    $pdo->commit();

} catch (Exception $e) {
    $pdo->rollBack();
    die("Database error");
}

// -----------------------
// Notify User (Telegram)
// -----------------------
sendTelegramMessage(
    $order['telegram_id'],
    "✅ <b>پرداخت با موفقیت انجام شد</b>\n\n" .
    "🧾 شماره سفارش: <b>#{$order_id}</b>\n" .
    "💰 مبلغ پرداختی: <b>{$order['total_price']} تومان</b>\n" .
    "🔖 کد رهگیری: <b>{$ref_id}</b>\n\n" .
    "⏳ سفارش شما ثبت شد و در صف انجام قرار گرفت.\n" .
    "پس از تکمیل، اطلاع‌رسانی خواهد شد."
);

// -----------------------
// Notify Admin
// -----------------------
if (defined('ADMIN_TELEGRAM_ID')) {
    sendTelegramMessage(
        ADMIN_TELEGRAM_ID,
        "📥 <b>پرداخت جدید</b>\n\n" .
        "🧾 سفارش: #{$order_id}\n" .
        "💰 مبلغ: {$order['total_price']} تومان\n" .
        "👤 کاربر: {$order['telegram_id']}\n" .
        "🔖 RefID: {$ref_id}"
    );
}

// -----------------------
// Final Output (Browser)
// -----------------------
echo "✅ پرداخت با موفقیت انجام شد.<br>می‌توانید به تلگرام بازگردید.";